


<?php
/* =====================================================================
   ANTIBOT SHIELD — SINGLE FILE (FINAL, STRAIGHTFORWARD)
   - Gate (server-side)
   - Inline Wall (silent/blank; keeps URL)
   - Admin panel (password + config + iplist manager + visitors)
   - IPLIST: downloads/overwrites ALL 9 files, cleans extras
   - DENY: shows article.php content (NO redirect)
   - Whitelist ALWAYS wins (even if also blocked)
   ===================================================================== */

/* ---------------------------------------------------------------------
   PATHS
   --------------------------------------------------------------------- */
$AB_LOG_FILE        = __DIR__ . '/visits.log';
$AB_WHITELIST_FILE  = __DIR__ . '/.whitelist';
$AB_BLOCKLIST_FILE  = __DIR__ . '/.blocklist';
$AB_CONFIG_FILE     = __DIR__ . '/antibot_config.json';
$AB_IPLIST_DIR      = __DIR__ . '/iplist';

/* ---------------------------------------------------------------------
   KEYS (put your real keys)
   --------------------------------------------------------------------- */
define('AB_IPQS_KEY',      'stF2iH9Vw5bcnHgewDa1YuXbA6e0qlva');
define('AB_PROXYCHECK_KEY','bced7133e0a31bfe80fc92a3b0bb4582aa4d7b4341fc3268901fbab01ad0d0d4');

/* ---------------------------------------------------------------------
   ADMIN
   --------------------------------------------------------------------- */
define('AB_ADMIN_PASSWORD', 'drainery');
define('AB_ADMIN_COOKIE',   'ab_admin');
define('AB_ADMIN_SALT',     'CHANGE_THIS_RANDOM_LONG_STRING_!_123456789');

/* ---------------------------------------------------------------------
   DENY TARGET (shown inline; not redirected)
   --------------------------------------------------------------------- */
define('AB_DENY_PAGE', __DIR__ . '/article.php');

/* ---------------------------------------------------------------------
   DEBUG (server-only page; wall remains silent)
   --------------------------------------------------------------------- */
$AB_DEBUG = false;

/* ---------------------------------------------------------------------
   IPLIST SOURCES (THE 9 FILES YOU WANT)
   - Update/Download overwrites each file.
   - Cleanup deletes any extra non-dot files in /iplist/ not in this list.
   --------------------------------------------------------------------- */
function ab_iplist_sources(): array {
    $av1 = 'https://raw.githubusercontent.com/antoinevastel/avastel-bot-ips-lists/refs/heads/master/avastel-proxy-bot-ips-1day.txt';
    $av5 = 'https://raw.githubusercontent.com/antoinevastel/avastel-bot-ips-lists/refs/heads/master/avastel-proxy-bot-ips-blocklist-5days.txt';
    $av8 = 'https://raw.githubusercontent.com/antoinevastel/avastel-bot-ips-lists/refs/heads/master/avastel-proxy-bot-ips-blocklist-8days.txt';

    return [
        // aliases (these 3 duplicate content but MUST exist as separate files per your request)
        'avastel_1day_alias' => ['name'=>'Avastel 1day alias', 'url'=>$av1, 'file'=>'avastel_1day.txt'],
        'avastel_5days_alias'=> ['name'=>'Avastel 5days alias', 'url'=>$av5, 'file'=>'avastel_5days.txt'],
        'avastel_8days_alias'=> ['name'=>'Avastel 8days alias', 'url'=>$av8, 'file'=>'avastel_8days.txt'],

        // originals
        'avastel_1day' => ['name'=>'Avastel 1day', 'url'=>$av1, 'file'=>'avastel-proxy-bot-ips-1day.txt'],
        'avastel_5days'=> ['name'=>'Avastel 5days', 'url'=>$av5, 'file'=>'avastel-proxy-bot-ips-blocklist-5days.txt'],
        'avastel_8days'=> ['name'=>'Avastel 8days', 'url'=>$av8, 'file'=>'avastel-proxy-bot-ips-blocklist-8days.txt'],

        'goodbots_all' => [
            'name' => 'GoodBots all.ips',
            'url'  => 'https://raw.githubusercontent.com/AnTheMaker/GoodBots/refs/heads/main/all.ips',
            'file' => 'goodbots_all.ips',
        ],
        'ipranges_v4' => [
            'name' => 'ipranges ipv4',
            'url'  => 'https://raw.githubusercontent.com/lord-alfred/ipranges/refs/heads/main/all/ipv4.txt',
            'file' => 'ipranges_ipv4.txt',
        ],
        'ipranges_v6' => [
            'name' => 'ipranges ipv6',
            'url'  => 'https://raw.githubusercontent.com/lord-alfred/ipranges/refs/heads/main/all/ipv6.txt',
            'file' => 'ipranges_ipv6.txt',
        ],
    ];
}

/* ---------------------------------------------------------------------
   CONFIG DEFAULTS
   --------------------------------------------------------------------- */
function ab_default_config(): array {
    return [
        'filters_enabled'       => true,    // false = log only (never deny)

        // Wall (client-side)
        'enable_tz_check'       => false,
        'enable_fp_wall'        => true,
        'enable_anti_headless'  => true,
        'allowed_timezones'     => [],       // empty = allow all

        // Countries
        'allowed_countries'     => [],       // empty = allow all (unless blocked)
        'blocked_countries'     => [],

        // APIs toggles
        'use_ipqs'              => true,
        'use_proxycheck'        => true,
        'use_ipn'               => true,

        // ProxyCheck: ONLY proxy field
        'proxycheck_only_proxy_field' => true,

        // IPQS fraud score optional
        'ipqs_use_fraud_score'  => false,
        'ipqs_fraud_threshold'  => 75,

        // Local lists
        'use_local_ip_lists'    => true,
        'iplist_enabled_files'  => array_values(array_map(fn($x)=>$x['file'], ab_iplist_sources())),

        // Param gate + sticky cookie
        'param_gate' => [
            'enabled'        => false,
            'param_name'     => 'id',
            'mode'           => 'off',   // off|exact|contains|file
            'exact_value'    => '',
            'contains_str'   => '',
            'values_file'    => 'values.txt',
            'remember_pass'  => true,
            'cookie_name'    => 'ab_entry_ok',
            'cookie_days'    => 30
        ],
    ];
}

function ab_load_config(string $file): array {
    $default = ab_default_config();
    if (!is_readable($file)) return $default;
    $j = json_decode(@file_get_contents($file), true);
    if (!is_array($j)) return $default;
    return array_replace_recursive($default, $j);
}

function ab_save_config(string $file, array $cfg): bool {
    return (bool)@file_put_contents(
        $file,
        json_encode($cfg, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES),
        LOCK_EX
    );
}

/* ---------------------------------------------------------------------
   HELPERS
   --------------------------------------------------------------------- */
function ab_e($s): string { return htmlspecialchars((string)$s, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8'); }

function ab_get_client_ip(): string {
    foreach (['HTTP_CF_CONNECTING_IP','HTTP_X_FORWARDED_FOR','REMOTE_ADDR'] as $h){
        if (!empty($_SERVER[$h])) {
            $first = explode(',', $_SERVER[$h])[0];
            return trim($first);
        }
    }
    return '0.0.0.0';
}

function ab_get_user_agent(): string {
    return $_SERVER['HTTP_USER_AGENT'] ?? '';
}

function ab_log_visit(string $file, array $data): void {
    @file_put_contents($file, implode(' | ', $data).PHP_EOL, FILE_APPEND | LOCK_EX);
}

function ab_parse_visit_line(string $line): array {
    $parts = explode(' | ', $line);
    $data  = ['json'=>'','ua'=>''];
    foreach($parts as $p){
        if (strpos($p,'JSON:')===0) $data['json'] = substr($p,5);
        elseif (strpos($p,'UA:')===0) $data['ua'] = substr($p,3);
    }
    return array_merge([
        'datetime'=>$parts[0]??'','ip'=>$parts[1]??'','decision'=>$parts[2]??'',
        'reason'=>$parts[3]??'','country'=>$parts[4]??'','region'=>$parts[5]??'',
        'isp'=>$parts[6]??'','org'=>$parts[7]??'','fraud'=>$parts[8]??'',
        'ua'=>$data['ua']??'','json'=>$data['json']
    ]);
}

function ab_device_type_from_ua(string $ua): string {
    $ua = strtolower($ua);
    if (preg_match('/android|iphone|ipad|ipod|blackberry|iemobile|opera mini|mobile/i',$ua)) return 'mobile';
    return 'desktop';
}

function ab_add_line_unique(string $file, string $line): bool {
    $line = trim($line);
    if ($line === '') return false;

    $lines = [];
    if (is_readable($file)){
        $lines = file($file, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
        foreach ($lines as $l){
            if (trim($l) === $line) return true;
        }
    }
    $lines[] = $line;
    return (bool)@file_put_contents($file, implode(PHP_EOL, $lines).PHP_EOL, LOCK_EX);
}

/* ---------------------------------------------------------------------
   CIDR MATCH (IPv4 + IPv6)
   --------------------------------------------------------------------- */
function ab_cidr_match(string $ip, string $cidr): bool {
    $ip = trim($ip);
    $cidr = trim($cidr);
    if ($ip === '' || $cidr === '' || strpos($cidr,'/') === false) return false;

    [$net, $mask] = explode('/', $cidr, 2);
    $mask = (int)$mask;

    $ipBin  = @inet_pton($ip);
    $netBin = @inet_pton($net);
    if ($ipBin === false || $netBin === false) return false;

    $ipLen = strlen($ipBin);
    if ($ipLen !== strlen($netBin)) return false;

    $maxBits = $ipLen * 8;
    if ($mask < 0 || $mask > $maxBits) return false;

    $bytes = intdiv($mask, 8);
    $bits  = $mask % 8;

    if ($bytes > 0){
        if (substr($ipBin, 0, $bytes) !== substr($netBin, 0, $bytes)) return false;
    }
    if ($bits === 0) return true;

    $ipByte  = ord($ipBin[$bytes]);
    $netByte = ord($netBin[$bytes]);
    $maskByte = (0xFF << (8 - $bits)) & 0xFF;

    return (($ipByte & $maskByte) === ($netByte & $maskByte));
}

/* ---------------------------------------------------------------------
   WHITELIST / BLOCKLIST
   - Whitelist ALWAYS wins (even if also blocked)
   --------------------------------------------------------------------- */
function ab_is_whitelisted(string $ip, string $whitelistFile): bool {
    if (!is_readable($whitelistFile)) return false;
    $lines = file($whitelistFile, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    foreach ($lines as $line){
        $line = trim($line);
        if ($line === '') continue;
        if (strcasecmp($line, $ip) === 0) return true;
    }
    return false;
}

function ab_is_blocked_local(string $ip, string $blockFile): bool {
    if (!is_readable($blockFile)) return false;
    $lines = file($blockFile, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    foreach ($lines as $line){
        $line = trim($line);
        if ($line === '') continue;

        if (strcasecmp($line, $ip) === 0) return true;
        if (strpos($line,'/') !== false){
            if (ab_cidr_match($ip, $line)) return true;
        }
    }
    return false;
}

/* ---------------------------------------------------------------------
   HTTP
   --------------------------------------------------------------------- */
function ab_http_get_json(string $url, int $timeout=6) {
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT        => $timeout,
        CURLOPT_CONNECTTIMEOUT => $timeout,
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_SSL_VERIFYHOST => false,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_MAXREDIRS      => 3,
        CURLOPT_USERAGENT      => 'Mozilla/5.0'
    ]);
    $body = curl_exec($ch);
    curl_close($ch);
    if (!$body) return null;
    $j = json_decode($body, true);
    return is_array($j) ? $j : null;
}

function ab_http_get_text(string $url, int $timeout=30): array {
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT        => $timeout,
        CURLOPT_CONNECTTIMEOUT => min(10, $timeout),
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_SSL_VERIFYHOST => false,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_MAXREDIRS      => 3,
        CURLOPT_USERAGENT      => 'Mozilla/5.0',
        CURLOPT_HTTPHEADER     => [
            'Cache-Control: no-cache',
            'Pragma: no-cache'
        ],
    ]);
    $body = curl_exec($ch);
    $err  = curl_error($ch);
    $code = (int)curl_getinfo($ch, CURLINFO_RESPONSE_CODE);
    curl_close($ch);
    return ['ok' => ($body !== false && $code >= 200 && $code < 300), 'code' => $code, 'err' => $err, 'body' => ($body === false ? '' : $body)];
}

/* ---------------------------------------------------------------------
   PARAM GATE (sticky cookie)
   --------------------------------------------------------------------- */
function ab_param_gate_allows(array $cfg, bool &$freshPass = false): bool {
    $freshPass = false;
    if (empty($cfg['param_gate']['enabled'])) return true;

    $pg = $cfg['param_gate'];

    // Sticky cookie bypass
    $remember = !empty($pg['remember_pass']);
    $cookieName = trim($pg['cookie_name'] ?? 'ab_entry_ok');
    if ($remember && $cookieName !== '' && !empty($_COOKIE[$cookieName]) && $_COOKIE[$cookieName] === '1'){
        return true;
    }

    $name = $pg['param_name'] ?? 'id';
    $mode = $pg['mode']       ?? 'off';
    $val  = $_GET[$name] ?? '';

    if ($mode === 'off') return true;

    if ($mode === 'exact'){
        $expected = $pg['exact_value'] ?? '';
        if ($expected === '') return false;
        $ok = hash_equals($expected, (string)$val);
        if ($ok) $freshPass = true;
        return $ok;
    }

    if ($mode === 'contains'){
        $needle = $pg['contains_str'] ?? '';
        if ($needle === '') return false;
        $ok = (strpos((string)$val, (string)$needle) !== false);
        if ($ok) $freshPass = true;
        return $ok;
    }

    if ($mode === 'file'){
        $file = $pg['values_file'] ?? 'values.txt';
        $path = __DIR__ . '/' . $file;
        if (!is_readable($path)) return false;
        $lines = file($path, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
        foreach ($lines as $line){
            $line = trim((string)$line);
            if ($line === '') continue;
            if (hash_equals($line, (string)$val)){
                $freshPass = true;
                return true;
            }
        }
        return false;
    }

    return true;
}

function ab_set_param_gate_cookie_if_needed(array $cfg): void {
    if (empty($cfg['param_gate']['enabled'])) return;
    $pg = $cfg['param_gate'];
    if (empty($pg['remember_pass'])) return;

    $cookieName = trim($pg['cookie_name'] ?? 'ab_entry_ok');
    if ($cookieName === '') return;

    $days = (int)($pg['cookie_days'] ?? 30);
    if ($days <= 0) $days = 30;

    $secure = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off');
    $expire = time() + ($days * 86400);

    @setcookie($cookieName, '1', [
        'expires'  => $expire,
        'path'     => '/',
        'secure'   => $secure,
        'httponly' => true,
        'samesite' => 'Lax'
    ]);
}

/* ---------------------------------------------------------------------
   INLINE WALL — SILENT/BLANK
   --------------------------------------------------------------------- */
function antibot_wall_inline(): void {
    global $AB_CONFIG_FILE;
    $cfg = ab_load_config($AB_CONFIG_FILE);

    $enableTZ  = !empty($cfg['enable_tz_check']);
    $enableFP  = !empty($cfg['enable_fp_wall']);
    $enableHL  = !empty($cfg['enable_anti_headless']);
    $allowedTZ = $cfg['allowed_timezones'] ?? [];

    $allowedTZJson = json_encode(array_values($allowedTZ), JSON_UNESCAPED_SLASHES | JSON_HEX_TAG | JSON_HEX_APOS | JSON_HEX_QUOT);

    header('Content-Type: text/html; charset=utf-8');
    ?>
<!doctype html><html><head>
<meta charset="utf-8">
<meta name="robots" content="noindex,nofollow">
<meta name="viewport" content="width=device-width,initial-scale=1">
<title></title>
<style>html,body{margin:0;padding:0;background:#fff}</style>
</head><body>
<script>
(function(){
  const ENABLE_TZ_CHECK      = <?= $enableTZ ? 'true' : 'false'; ?>;
  const ENABLE_FP_WALL       = <?= $enableFP ? 'true' : 'false'; ?>;
  const ENABLE_ANTI_HEADLESS = <?= $enableHL ? 'true' : 'false'; ?>;
  const ALLOWED_TIMEZONES    = <?= $allowedTZJson ?: '[]'; ?>;

  function pass(){
    try{
      document.cookie="wall_ok=1; path=/; max-age=86400; SameSite=Lax";
      if(ENABLE_TZ_CHECK) document.cookie="tz_passed=1; path=/; max-age=86400; SameSite=Lax";
    }catch(e){}
    try{ location.reload(); }catch(e){ location.href=location.href; }
  }

  function deny(){
    // server will show article.php; do nothing here to stay silent
    try{ location.reload(); }catch(e){ location.href=location.href; }
  }

  function headless(){
    if(!ENABLE_ANTI_HEADLESS) return false;
    try{
      if(navigator.webdriver) return true;
      const p=(navigator.plugins||[]).length;
      const l=(navigator.languages||[]).length;
      const h=navigator.hardwareConcurrency||0;
      if(p===0 && l<=1 && h<=2) return true;
      return false;
    }catch(e){ return true; }
  }

  function fp(){
    if(!ENABLE_FP_WALL) return false;
    try{
      let score=0;
      const hasTouch=('ontouchstart' in window) || (navigator.maxTouchPoints>0);
      const ua=navigator.userAgent||"";
      const isMobile=/Android|webOS|iPhone|iPad|iPod|BlackBerry|IEMobile|Opera Mini/i.test(ua);
      if(isMobile && !hasTouch) score+=2;
      const lang=(navigator.language||"").toLowerCase();
      if(!lang) score+=1;
      const lc=(navigator.languages||[]).length;
      if(lc===0) score+=1;
      return score>=3;
    }catch(e){ return true; }
  }

  function tzok(){
    if(!ENABLE_TZ_CHECK) return true;
    try{
      const tz=(Intl && Intl.DateTimeFormat) ? (Intl.DateTimeFormat().resolvedOptions().timeZone||"Unavailable") : "Unavailable";
      if(!Array.isArray(ALLOWED_TIMEZONES) || ALLOWED_TIMEZONES.length===0) return true;
      return tz!=="Unavailable" && ALLOWED_TIMEZONES.includes(tz);
    }catch(e){ return false; }
  }

  try{
    if(headless()) return deny();
    if(fp()) return deny();
    if(!tzok()) return deny();
    return pass();
  }catch(e){
    return deny();
  }
})();
</script>
</body></html>
<?php
}

/* ---------------------------------------------------------------------
   IPLIST — parsing helpers (NO CACHE/COMPILE)
   --------------------------------------------------------------------- */
function ab_iplist_dir_ensure(string $dir): bool {
    if (is_dir($dir)) return true;
    return @mkdir($dir, 0755, true);
}

function ab_iplist_path(string $dir, string $filename): string {
    return rtrim($dir, '/').'/'.ltrim($filename, '/');
}

function ab_iplist_extract_token(string $line): string {
    $line = trim($line);
    if ($line === '') return '';
    if ($line[0] === '#') return '';
    if (stripos($line, 'ip_address;') === 0) return '';
    // first column if ';' exists
    $first = explode(';', $line, 2)[0];
    return trim($first);
}

function ab_iplist_is_ip_or_cidr(string $token): bool {
    if ($token === '') return false;
    if (strpos($token, '/') !== false){
        [$net,$mask] = explode('/', $token, 2);
        if (@inet_pton($net) === false) return false;
        if (!is_numeric($mask)) return false;
        return true;
    }
    return (@inet_pton($token) !== false);
}

function ab_iplist_file_matches(string $ip, string $path): bool {
    if (!is_readable($path)) return false;
    $fh = new SplFileObject($path, 'r');
    while (!$fh->eof()){
        $tok = ab_iplist_extract_token($fh->fgets());
        if ($tok === '' || !ab_iplist_is_ip_or_cidr($tok)) continue;

        if (strpos($tok,'/') !== false){
            if (ab_cidr_match($ip, $tok)) return true;
        } else {
            if (strcasecmp($tok, $ip) === 0) return true;
        }
    }
    return false;
}

function ab_iplist_matches(string $ip, string $iplistDir, array $cfg): bool {
    if (empty($cfg['use_local_ip_lists'])) return false;

    $enabledFiles = $cfg['iplist_enabled_files'] ?? [];
    if (!is_array($enabledFiles) || empty($enabledFiles)){
        // if empty in config -> treat as all files enabled
        $enabledFiles = array_values(array_map(fn($x)=>$x['file'], ab_iplist_sources()));
    }

    foreach ($enabledFiles as $file){
        $file = trim((string)$file);
        if ($file === '') continue;
        $path = ab_iplist_path($iplistDir, $file);
        if (ab_iplist_file_matches($ip, $path)) return true;
    }
    return false;
}

/* ---------------------------------------------------------------------
   DENY RENDER (SHOW article.php CONTENT, NO REDIRECT)
   --------------------------------------------------------------------- */
function ab_deny_show_and_exit(): void {
    http_response_code(403);
    if (is_file(AB_DENY_PAGE)) {
        include AB_DENY_PAGE;
    }
    exit;
}

/* ---------------------------------------------------------------------
   DEBUG PAGE (server-side only)
   --------------------------------------------------------------------- */
function ab_render_debug_page($decision,$reason,$country,$isp,$org,$notes,$ipqs,$pc,$ipn){
    header('Content-Type: text/html; charset=utf-8'); ?>
<!doctype html>
<html><head><meta charset="utf-8"><title>antibot DEBUG</title>
<style>
body{font-family:system-ui;background:#0e0e0e;color:#eee;padding:20px}
pre{background:#111;color:#8ff;padding:15px;border-radius:8px;font-size:13px;white-space:pre-wrap}
.allow{color:#7ef08a} .deny{color:#ff8b8b}
</style></head><body>
<h1>ANTIBOT DEBUG</h1>
<h2>Decision: <?= $decision==='allowed' ? '<span class="allow">ALLOWED</span>' : '<span class="deny">DENIED</span>' ?></h2>
<p><b>Reason:</b> <?=ab_e($reason)?></p>
<p><b>Country:</b> <?=ab_e($country)?></p>
<p><b>ISP:</b> <?=ab_e($isp)?></p>
<p><b>Org:</b> <?=ab_e($org)?></p>
<h2>Notes</h2><pre><?php foreach((array)$notes as $n) echo $n."\n"; ?></pre>
<h2>IPQS</h2><pre><?=ab_e(print_r($ipqs,true))?></pre>
<h2>ProxyCheck</h2><pre><?=ab_e(print_r($pc,true))?></pre>
<h2>i.pn</h2><pre><?=ab_e(print_r($ipn,true))?></pre>
</body></html>
<?php }

/* ---------------------------------------------------------------------
   GATE: MAIN SHIELD LOGIC
   --------------------------------------------------------------------- */
function antibot_gate(){
    global $AB_LOG_FILE, $AB_WHITELIST_FILE, $AB_BLOCKLIST_FILE, $AB_CONFIG_FILE;
    global $AB_IPLIST_DIR, $AB_DEBUG;

    $cfg = ab_load_config($AB_CONFIG_FILE);

    $ip  = ab_get_client_ip();
    $ua  = ab_get_user_agent();
    $now = date('Y-m-d H:i:s');

    $decision = 'allowed';
    $reason   = 'clean';
    $country  = '';
    $region   = '';
    $isp      = '';
    $org      = '';
    $fraud    = 0;

    $ipqs = $pc = $ipn = null;
    $notes = [];

    ab_iplist_dir_ensure($AB_IPLIST_DIR);

    /* 0) WHITELIST ALWAYS WINS */
    if (ab_is_whitelisted($ip, $AB_WHITELIST_FILE)){
        ab_log_visit($AB_LOG_FILE, [
            $now, $ip, 'allowed', 'whitelisted', $country, $region, $isp, $org, $fraud,
            "UA:$ua",
            "JSON:{}"
 
        ]);
        return;
    }

    /* 1) If blocking OFF => log only and allow */
    if (empty($cfg['filters_enabled'])) {
        ab_log_visit($AB_LOG_FILE, [
            $now, $ip, 'allowed', 'clean', $country, $region, $isp, $org, $fraud,
            "UA:$ua",
            "JSON:{}"

        ]);
        return;
    }

    /* 2) Param gate (sticky cookie) */
    $paramGateFreshPass = false;
    if (!ab_param_gate_allows($cfg, $paramGateFreshPass)){
        ab_log_visit($AB_LOG_FILE, [
            $now, $ip, 'denied', 'param_gate', $country, $region, $isp, $org, $fraud,
            "UA:$ua",
            "JSON:{}"
        ]);
        if ($AB_DEBUG) { ab_render_debug_page('denied','param_gate',$country,$isp,$org,$notes,$ipqs,$pc,$ipn); exit; }
        ab_deny_show_and_exit();
    }

    /* 3) Local manual blocklist */
    if (ab_is_blocked_local($ip, $AB_BLOCKLIST_FILE)){
        ab_log_visit($AB_LOG_FILE, [
            $now, $ip, 'denied', 'local_blocklist', $country, $region, $isp, $org, $fraud,
            "UA:$ua",
            "JSON:{}"
        ]);
        if ($AB_DEBUG) { ab_render_debug_page('denied','local_blocklist',$country,$isp,$org,$notes,$ipqs,$pc,$ipn); exit; }
        ab_deny_show_and_exit();
    }

    /* 4) Local iplist folder (the 9 downloaded files) */
    if (!empty($cfg['use_local_ip_lists'])){
        if (ab_iplist_matches($ip, $AB_IPLIST_DIR, $cfg)){
            ab_log_visit($AB_LOG_FILE, [
                $now, $ip, 'denied', 'iplist_blocked', $country, $region, $isp, $org, $fraud,
                "UA:$ua",
                "JSON:{}"
            ]);
            if ($AB_DEBUG) { ab_render_debug_page('denied','iplist_blocked',$country,$isp,$org,$notes,$ipqs,$pc,$ipn); exit; }
            ab_deny_show_and_exit();
        }
    }

    /* 5) Inline wall (silent/blank) */
    $needWall = false;
    if (!empty($cfg['enable_tz_check']) && (empty($_COOKIE['tz_passed']) || $_COOKIE['tz_passed'] !== '1')) $needWall = true;
    if (!empty($cfg['enable_fp_wall']) && (empty($_COOKIE['wall_ok']) || $_COOKIE['wall_ok'] !== '1')) $needWall = true;
    if ($needWall){
        antibot_wall_inline();
        exit;
    }

    /* 6) Reputation APIs */
    // IPQS
    if (!empty($cfg['use_ipqs']) && AB_IPQS_KEY){
        $ipqs_url = "https://ipqualityscore.com/api/json/ip/".AB_IPQS_KEY."/".$ip;
        $ipqs = ab_http_get_json($ipqs_url);
        if (is_array($ipqs) && !empty($ipqs['success'])){
            $fraud   = (int)($ipqs['fraud_score'] ?? 0);
            $country = strtoupper((string)($ipqs['country_code'] ?? $country));
            $region  = (string)($ipqs['region'] ?? $region);
            $isp     = (string)($ipqs['ISP'] ?? $isp);
            $org     = (string)($ipqs['organization'] ?? $org);

            foreach(['active_tor','active_vpn','proxy','vpn','tor','recent_abuse','bot_status','is_crawler'] as $f){
                if (!empty($ipqs[$f])){
                    ab_log_visit($AB_LOG_FILE, [
                        $now, $ip, 'denied', "ipqs_$f", $country, $region, $isp, $org, $fraud,
                        "UA:$ua",
                        "JSON:".json_encode(['ipqs'=>$ipqs], JSON_UNESCAPED_SLASHES)
                    ]);
                    if ($AB_DEBUG) { ab_render_debug_page('denied',"ipqs_$f",$country,$isp,$org,$notes,$ipqs,$pc,$ipn); exit; }
                    ab_deny_show_and_exit();
                }
            }

            if (!empty($cfg['ipqs_use_fraud_score'])){
                $thr = (int)($cfg['ipqs_fraud_threshold'] ?? 75);
                if ($thr < 0) $thr = 0;
                if ($thr > 100) $thr = 100;
                if ($fraud >= $thr){
                    ab_log_visit($AB_LOG_FILE, [
                        $now, $ip, 'denied', "ipqs_fraud_score", $country, $region, $isp, $org, $fraud,
                        "UA:$ua",
                        "JSON:".json_encode(['ipqs'=>$ipqs], JSON_UNESCAPED_SLASHES)
                    ]);
                    if ($AB_DEBUG) { ab_render_debug_page('denied',"ipqs_fraud_score",$country,$isp,$org,$notes,$ipqs,$pc,$ipn); exit; }
                    ab_deny_show_and_exit();
                }
            }
        }
    }

    // ProxyCheck (ONLY proxy field)
    if (!empty($cfg['use_proxycheck']) && AB_PROXYCHECK_KEY){
        $pc_url = "https://proxycheck.io/v2/".$ip."?key=".AB_PROXYCHECK_KEY."&asn=1&vpn=1";
        $pc = ab_http_get_json($pc_url);
        if (is_array($pc) && ($pc['status'] ?? '') === 'ok' && isset($pc[$ip])){
            $e = $pc[$ip];
            $country = strtoupper((string)($e['isocode'] ?? $country));
            $isp     = (string)($e['provider'] ?? $isp);
            $org     = (string)($e['organisation'] ?? $org);

            $proxyVal = strtolower((string)($e['proxy'] ?? ''));
            if (!empty($cfg['proxycheck_only_proxy_field'])){
                if ($proxyVal !== 'no'){
                    ab_log_visit($AB_LOG_FILE, [
                        $now, $ip, 'denied', "proxycheck_proxy", $country, $region, $isp, $org, $fraud,
                        "UA:$ua",
                        "JSON:".json_encode(['pc'=>$pc], JSON_UNESCAPED_SLASHES)
                    ]);
                    if ($AB_DEBUG) { ab_render_debug_page('denied',"proxycheck_proxy",$country,$isp,$org,$notes,$ipqs,$pc,$ipn); exit; }
                    ab_deny_show_and_exit();
                }
            }
        }
    }

    // i.pn
    if (!empty($cfg['use_ipn'])){
        $ipn_url = "https://i.pn/json/".$ip;
        $ipn = ab_http_get_json($ipn_url);
        if (is_array($ipn) && ($ipn['status'] ?? '') === 'success'){
            if (!empty($ipn['proxy']) || !empty($ipn['hosting'])){
                ab_log_visit($AB_LOG_FILE, [
                    $now, $ip, 'denied', "ipn_proxyhosting", $country, $region, $isp, $org, $fraud,
                    "UA:$ua",
                    "JSON:".json_encode(['ipn'=>$ipn], JSON_UNESCAPED_SLASHES)
                ]);
                if ($AB_DEBUG) { ab_render_debug_page('denied',"ipn_proxyhosting",$country,$isp,$org,$notes,$ipqs,$pc,$ipn); exit; }
                ab_deny_show_and_exit();
            }
        }
    }

    /* 7) Country allow/block logic
       - allowed empty => allow all (unless blocked)
    */
    $allowed = $cfg['allowed_countries'] ?? [];
    $blocked = $cfg['blocked_countries'] ?? [];

    if (is_array($allowed) && !empty($allowed)){
        if (!in_array($country, $allowed, true)){
            ab_log_visit($AB_LOG_FILE, [
                $now, $ip, 'denied', "country_not_allowed", $country, $region, $isp, $org, $fraud,
                "UA:$ua",
                "JSON:{}"
            ]);
            if ($AB_DEBUG) { ab_render_debug_page('denied',"country_not_allowed",$country,$isp,$org,$notes,$ipqs,$pc,$ipn); exit; }
            ab_deny_show_and_exit();
        }
    }

    if (is_array($blocked) && !empty($blocked)){
        if (in_array($country, $blocked, true)){
            ab_log_visit($AB_LOG_FILE, [
                $now, $ip, 'denied', "country_blocked", $country, $region, $isp, $org, $fraud,
                "UA:$ua",
                "JSON:{}"
            ]);
            if ($AB_DEBUG) { ab_render_debug_page('denied',"country_blocked",$country,$isp,$org,$notes,$ipqs,$pc,$ipn); exit; }
            ab_deny_show_and_exit();
        }
    }

    /* 8) Allowed -> log + sticky cookie if fresh pass */
    ab_log_visit($AB_LOG_FILE, [
        $now, $ip, 'allowed', ($paramGateFreshPass ? 'clean_entry' : 'clean'),
        $country, $region, $isp, $org, $fraud,
        "UA:$ua",
        "JSON:".json_encode(['ipqs'=>$ipqs,'pc'=>$pc,'ipn'=>$ipn], JSON_UNESCAPED_SLASHES)
    ]);

    if ($paramGateFreshPass){
        ab_set_param_gate_cookie_if_needed($cfg);
    }
}

/* ---------------------------------------------------------------------
   ADMIN AUTH
   --------------------------------------------------------------------- */
function ab_admin_cookie_value(): string {
    return hash_hmac('sha256', AB_ADMIN_PASSWORD, AB_ADMIN_SALT);
}
function ab_is_admin_authed(): bool {
    $v = $_COOKIE[AB_ADMIN_COOKIE] ?? '';
    return is_string($v) && hash_equals(ab_admin_cookie_value(), $v);
}
function ab_set_admin_cookie(): void {
    $secure = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off');
    $expire = time() + (86400 * 3650);
    @setcookie(AB_ADMIN_COOKIE, ab_admin_cookie_value(), [
        'expires'  => $expire,
        'path'     => '/',
        'secure'   => $secure,
        'httponly' => true,
        'samesite' => 'Lax'
    ]);
}
function ab_clear_admin_cookie(): void {
    $secure = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off');
    @setcookie(AB_ADMIN_COOKIE, '', [
        'expires'  => time() - 3600,
        'path'     => '/',
        'secure'   => $secure,
        'httponly' => true,
        'samesite' => 'Lax'
    ]);
}

/* ---------------------------------------------------------------------
   ADMIN: IPLIST API
   - download(key) overwrites file always
   - cleanup removes extra files not in the 9 file list
   --------------------------------------------------------------------- */
function ab_admin_iplist_api(){
    global $AB_IPLIST_DIR;

    if (!ab_is_admin_authed()){
        http_response_code(403);
        header('Content-Type: application/json; charset=utf-8');
        echo json_encode(['ok'=>false,'error'=>'forbidden']);
        exit;
    }

    ab_iplist_dir_ensure($AB_IPLIST_DIR);
    $sources = ab_iplist_sources();

    $action = $_GET['action'] ?? 'list';
    header('Content-Type: application/json; charset=utf-8');

    if ($action === 'list'){
        $items = [];
        foreach ($sources as $k=>$s){
            $path = ab_iplist_path($AB_IPLIST_DIR, $s['file']);
            $items[] = [
                'key'  => $k,
                'name' => $s['name'],
                'url'  => $s['url'],
                'file' => $s['file'],
                'exists' => is_file($path),
                'size' => is_file($path) ? (int)@filesize($path) : 0,
                'mtime' => is_file($path) ? (int)@filemtime($path) : 0,
            ];
        }
        echo json_encode(['ok'=>true,'items'=>$items], JSON_UNESCAPED_SLASHES);
        exit;
    }

    if ($action === 'download'){
        $key = $_GET['key'] ?? '';
        if ($key === '' || !isset($sources[$key])){
            echo json_encode(['ok'=>false,'error'=>'invalid_key']);
            exit;
        }

        $s = $sources[$key];
        $url  = $s['url'];
        $file = $s['file'];
        $path = ab_iplist_path($AB_IPLIST_DIR, $file);

        $r = ab_http_get_text($url, 45);
        if (!$r['ok']){
            echo json_encode([
                'ok'=>false,
                'key'=>$key,
                'file'=>$file,
                'error'=>"download_failed",
                'http_code'=>$r['code'],
                'curl_error'=>$r['err'],
                'manual'=>"Download manually from the URL and place it into /iplist/{$file}"
            ], JSON_UNESCAPED_SLASHES);
            exit;
        }

        $tmp = $path.'.tmp';
        $bytes = @file_put_contents($tmp, $r['body'], LOCK_EX);
        if ($bytes === false){
            @unlink($tmp);
            echo json_encode([
                'ok'=>false,
                'key'=>$key,
                'file'=>$file,
                'error'=>"write_failed",
                'manual'=>"Fix permissions on /iplist/ then download manually to /iplist/{$file}"
            ], JSON_UNESCAPED_SLASHES);
            exit;
        }

        // Overwrite ALWAYS (delete old then rename)
        if (is_file($path)) @unlink($path);
        @rename($tmp, $path);
        @chmod($path, 0644);

        echo json_encode([
            'ok'=>true,
            'key'=>$key,
            'file'=>$file,
            'size'=>(int)@filesize($path),
            'mtime'=>(int)@filemtime($path)
        ], JSON_UNESCAPED_SLASHES);
        exit;
    }

    if ($action === 'cleanup'){
        $keep = [];
        foreach ($sources as $s){
            $keep[$s['file']] = true;
        }

        $deleted = [];
        $dh = @opendir($AB_IPLIST_DIR);
        if ($dh){
            while (($f = readdir($dh)) !== false){
                if ($f === '.' || $f === '..') continue;
                if ($f[0] === '.') continue; // keep dotfiles
                $full = ab_iplist_path($AB_IPLIST_DIR, $f);
                if (!is_file($full)) continue;
                if (!empty($keep[$f])) continue;

                if (@unlink($full)){
                    $deleted[] = $f;
                }
            }
            closedir($dh);
        }

        echo json_encode(['ok'=>true,'deleted'=>$deleted], JSON_UNESCAPED_SLASHES);
        exit;
    }

    echo json_encode(['ok'=>false,'error'=>'unknown_action']);
    exit;
}

/* ---------------------------------------------------------------------
   ADMIN PANEL
   --------------------------------------------------------------------- */
function antibot_admin(){
    global $AB_LOG_FILE, $AB_WHITELIST_FILE, $AB_BLOCKLIST_FILE, $AB_CONFIG_FILE, $AB_IPLIST_DIR;

    if (isset($_GET['iplist_api'])){
        ab_admin_iplist_api();
        exit;
    }

    if (isset($_GET['logout'])){
        ab_clear_admin_cookie();
        header("Location: ".basename(__FILE__)."?admin=1");
        exit;
    }

    // Login
    if (!ab_is_admin_authed()){
        $err = '';
        if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['admin_pass'])){
            if (hash_equals(AB_ADMIN_PASSWORD, (string)$_POST['admin_pass'])){
                ab_set_admin_cookie();
                header("Location: ".basename(__FILE__)."?admin=1");
                exit;
            } else {
                $err = 'Wrong password.';
            }
        }

        header('Content-Type: text/html; charset=utf-8');
        ?>
<!doctype html>
<html><head><meta charset="utf-8"><meta name="viewport" content="width=device-width,initial-scale=1">
<title>Admin Login</title>
<style>
body{background:#05070b;color:#e2f3ff;font-family:system-ui;margin:0;padding:24px}
.card{max-width:420px;margin:70px auto;background:#0b1118;border:1px solid rgba(255,255,255,.08);
border-radius:14px;padding:16px 16px 18px;box-shadow:0 12px 40px rgba(0,0,0,.5)}
label{font-size:12px;color:#9fb6cd}
input{width:100%;padding:10px 12px;border-radius:10px;border:1px solid rgba(255,255,255,.08);
background:#05070b;color:#e2f3ff;font-size:14px;margin-top:6px}
button{margin-top:12px;width:100%;padding:10px 12px;border:0;border-radius:10px;
background:#1fb6ff2b;border:1px solid #1fb6ff;color:#eff8ff;cursor:pointer;font-size:14px}
.err{margin-top:10px;color:#ff9fb0;font-size:13px}
</style></head><body>
<div class="card">
  <h2 style="margin:0 0 10px 0;">Antibot Admin</h2>
  <form method="post">
    <label>Password</label>
    <input type="password" name="admin_pass" autofocus>
    <button type="submit">Login</button>
    <?php if($err): ?><div class="err"><?=ab_e($err)?></div><?php endif; ?>
  </form>
</div>
</body></html>
<?php
        exit;
    }

    ab_iplist_dir_ensure($AB_IPLIST_DIR);

    $cfg = ab_load_config($AB_CONFIG_FILE);
    $message = '';
    $msgType = 'ok';

    // Reset
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($_POST['action'] ?? '') === 'reset_config'){
        $cfg = ab_default_config();
        if (ab_save_config($AB_CONFIG_FILE, $cfg)){
            $message = 'Configuration reset to defaults.';
            $msgType = 'ok';
        } else {
            $message = 'Failed to reset configuration.';
            $msgType = 'err';
        }
    }

    // POST actions
    if ($_SERVER['REQUEST_METHOD'] === 'POST'){
        $action = $_POST['action'] ?? '';

        if (in_array($action,['block_ip','whitelist'],true)){
            $ip = trim($_POST['ip'] ?? '');
            if ($ip === ''){
                $message = "No IP/CIDR provided.";
                $msgType = 'err';
            } else {
                if ($action === 'block_ip'){
                    $ok = ab_add_line_unique($AB_BLOCKLIST_FILE, $ip);
                    $message = $ok ? "Blocked: $ip" : "Failed (or exists): $ip";
                    $msgType = $ok ? 'ok' : 'err';
                } elseif ($action === 'whitelist'){
                    // whitelist always wins; do not touch blocklist
                    $ok = ab_add_line_unique($AB_WHITELIST_FILE, $ip);
                    $message = $ok ? "Whitelisted: $ip" : "Failed (or exists): $ip";
                    $msgType = $ok ? 'ok' : 'err';
                }
            }
        }

        if ($action === 'save_config'){
            $cfg['filters_enabled']      = !empty($_POST['filters_enabled']);
            $cfg['enable_tz_check']      = !empty($_POST['enable_tz_check']);
            $cfg['enable_fp_wall']       = !empty($_POST['enable_fp_wall']);
            $cfg['enable_anti_headless'] = !empty($_POST['enable_anti_headless']);

            $cfg['use_ipqs']        = !empty($_POST['use_ipqs']);
            $cfg['use_proxycheck']  = !empty($_POST['use_proxycheck']);
            $cfg['use_ipn']         = !empty($_POST['use_ipn']);

            $cfg['proxycheck_only_proxy_field'] = !empty($_POST['proxycheck_only_proxy_field']);

            $cfg['ipqs_use_fraud_score'] = !empty($_POST['ipqs_use_fraud_score']);
            $cfg['ipqs_fraud_threshold'] = (int)($_POST['ipqs_fraud_threshold'] ?? 75);

            $cfg['use_local_ip_lists'] = !empty($_POST['use_local_ip_lists']);

            // enabled iplist files checkboxes
            $enabled_files = $_POST['iplist_enabled_files'] ?? [];
            if (!is_array($enabled_files)) $enabled_files = [];
            $cfg['iplist_enabled_files'] = array_values(array_unique(array_map('trim',$enabled_files)));

            // TZ multiselect
            $allowed_tz = $_POST['allowed_timezones'] ?? [];
            if (!is_array($allowed_tz)) $allowed_tz = [];
            $allowed_tz = array_values(array_unique(array_map('trim',$allowed_tz)));
            $cfg['allowed_timezones'] = $allowed_tz;

            // Countries CSV
            $allowed_countries = strtoupper(trim($_POST['allowed_countries'] ?? ''));
            $blocked_countries = strtoupper(trim($_POST['blocked_countries'] ?? ''));
            $cfg['allowed_countries'] = $allowed_countries === '' ? [] :
                array_values(array_filter(array_map('trim', explode(',', $allowed_countries))));
            $cfg['blocked_countries'] = $blocked_countries === '' ? [] :
                array_values(array_filter(array_map('trim', explode(',', $blocked_countries))));

            // Param gate
            $cfg['param_gate']['enabled']        = !empty($_POST['param_gate_enabled']);
            $cfg['param_gate']['param_name']     = trim($_POST['param_name'] ?? 'id');
            $cfg['param_gate']['mode']           = $_POST['param_mode'] ?? 'off';
            $cfg['param_gate']['exact_value']    = trim($_POST['param_exact_value'] ?? '');
            $cfg['param_gate']['contains_str']   = trim($_POST['param_contains_str'] ?? '');
            $cfg['param_gate']['values_file']    = trim($_POST['param_values_file'] ?? 'values.txt');
            $cfg['param_gate']['remember_pass']  = !empty($_POST['param_remember_pass']);
            $cfg['param_gate']['cookie_name']    = trim($_POST['param_cookie_name'] ?? 'ab_entry_ok');
            $cfg['param_gate']['cookie_days']    = (int)($_POST['param_cookie_days'] ?? 30);
            if ($cfg['param_gate']['cookie_days'] <= 0) $cfg['param_gate']['cookie_days'] = 30;
            if ($cfg['param_gate']['cookie_name'] === '') $cfg['param_gate']['cookie_name'] = 'ab_entry_ok';

            if (ab_save_config($AB_CONFIG_FILE,$cfg)){
                $message = ($message ? $message.' ' : '').'Configuration saved.';
                $msgType = 'ok';
            } else {
                $message = ($message ? $message.' ' : '').'Failed to save configuration.';
                $msgType = 'err';
            }
        }
    }

    // logs
    $lines = file_exists($AB_LOG_FILE)
        ? array_reverse(file($AB_LOG_FILE, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES))
        : [];

    $statusFilter  = $_GET['status']  ?? 'all';
    $countryFilter = strtoupper(trim($_GET['country'] ?? ''));
    $deviceFilter  = $_GET['device']  ?? 'all';

    $allTimezones = DateTimeZone::listIdentifiers();
    $sources = ab_iplist_sources();

    // stats
    $total = $allowed = $denied = 0;
    foreach ($lines as $line){
        $p = ab_parse_visit_line($line);
        if (!$p['ip']) continue;
        $total++;
        if (($p['decision'] ?? '') === 'denied') $denied++;
        else $allowed++;
    }

    header('Content-Type: text/html; charset=utf-8');
    ?>
<!doctype html>
<html>
<head>
<meta charset="utf-8">
<title>Anti-Bot Visitors</title>
<meta name="viewport" content="width=device-width,initial-scale=1">
<style>
:root{--bg:#05070b;--panel:#0b1118;--soft:#101822;--b:rgba(255,255,255,.08);--t:#e2f3ff;--ts:#9fb6cd;--acc:#1fb6ff;--acc2:#1fb6ff2b;--ok:#27d17d;--ok2:#27d17d1a;--bad:#ff4d6d;--bad2:#ff4d6d1a;}
*{box-sizing:border-box}
body{margin:0;padding:16px;background:radial-gradient(circle at top left,#17212b,#05070b 55%);color:var(--t);font-family:system-ui}
h1{margin:0 0 6px 0;font-size:20px}
.sub{color:var(--ts);font-size:12px;margin-bottom:12px}
.row{display:flex;gap:10px;flex-wrap:wrap;align-items:center}
.pill{display:inline-flex;gap:8px;align-items:center;padding:5px 10px;border:1px solid var(--b);border-radius:999px;background:rgba(0,0,0,.35);font-size:11px;color:var(--ts)}
.pill b{color:var(--t)}
.btn{padding:6px 12px;border-radius:999px;border:1px solid var(--b);background:var(--soft);color:var(--ts);cursor:pointer;font-size:11px}
.btn.primary{background:var(--acc2);border-color:var(--acc);color:#eff8ff}
.btn.danger{border-color:var(--bad);color:var(--bad)}
.btn.success{border-color:var(--ok);color:var(--ok)}
.btn.small{padding:4px 9px;font-size:10px}
.panel{background:var(--panel);border:1px solid var(--b);border-radius:14px;padding:12px 14px;box-shadow:0 12px 40px rgba(0,0,0,.5)}
.grid{display:grid;grid-template-columns:minmax(0,1.35fr) minmax(0,1fr);gap:12px;margin:12px 0}
@media(max-width:1050px){.grid{grid-template-columns:1fr}}
label{font-size:11px;color:var(--ts)}
input[type=text],select{width:100%;background:#05070b;border:1px solid var(--b);border-radius:8px;color:var(--t);padding:6px 8px;font-size:12px}
select[multiple]{min-height:120px}
.toggle{margin-bottom:6px}
.toggle label{font-size:12px}
.toggle input{margin-right:6px;transform:scale(.95)}
.msg{padding:8px 11px;border:1px solid var(--b);border-radius:8px;font-size:12px;margin:10px 0}
.msg.ok{border-color:var(--ok);background:var(--ok2);color:#caffdb}
.msg.err{border-color:var(--bad);background:var(--bad2);color:#ffd5dd}
.tablewrap{margin-top:10px;border:1px solid var(--b);border-radius:10px;overflow:auto;background:var(--soft)}
table{border-collapse:collapse;min-width:980px;width:100%;font-size:12px}
th,td{padding:7px 8px;border-bottom:1px solid rgba(255,255,255,.04);vertical-align:top}
th{color:var(--ts);text-transform:uppercase;font-size:11px;letter-spacing:.08em;background:rgba(0,0,0,.25)}
.badge{padding:3px 7px;border-radius:999px;font-size:10px;text-transform:uppercase;letter-spacing:.08em;border:1px solid rgba(255,255,255,.08)}
.badge.ok{background:var(--ok2);color:var(--ok);border-color:rgba(39,209,125,.4)}
.badge.bad{background:var(--bad2);color:var(--bad);border-color:rgba(255,77,109,.4)}
td.ua,td.reason{max-width:260px;overflow:hidden;text-overflow:ellipsis;white-space:nowrap}
td.ua:hover,td.reason:hover{white-space:normal;max-width:380px}
details summary{cursor:pointer;color:var(--acc);font-size:11px}
pre{background:#05080b;padding:8px;border-radius:8px;max-height:260px;overflow:auto;color:#a7e8ff;font-size:11px}
.smalltxt{font-size:11px;color:var(--ts)}
hr{border:0;border-top:1px solid rgba(255,255,255,.08);margin:10px 0}
.prog{height:10px;background:rgba(255,255,255,.08);border-radius:999px;overflow:hidden}
.prog > div{height:100%;width:0;background:rgba(31,182,255,.7)}
.list{margin:8px 0 0 0;padding:0 0 0 18px;font-size:12px;color:var(--ts)}
.flag{width:18px;height:18px;border-radius:4px;vertical-align:middle;margin-right:6px}
</style>
</head>
<body>

<div class="row" style="justify-content:space-between">
  <div>
    <h1>Anti-Bot Visitors</h1>
    <div class="sub">Admin URL: <code><?=ab_e(basename(__FILE__))?>?admin=1</code></div>
  </div>
  <div class="row">
    <span class="pill">Blocking: <b><?= !empty($cfg['filters_enabled']) ? 'ON' : 'OFF (log only)' ?></b></span>
    <a class="btn" href="<?=ab_e(basename(__FILE__))?>?admin=1&logout=1">Logout</a>
  </div>
</div>

<div class="row" style="margin-top:10px">
  <span class="pill">Total <b><?= (int)$total ?></b></span>
  <span class="pill">Allowed <b><?= (int)$allowed ?></b></span>
  <span class="pill">Denied <b><?= (int)$denied ?></b></span>
</div>

<?php if($message): ?>
  <div class="msg <?= $msgType==='err'?'err':'ok' ?>"><?=ab_e($message)?></div>
<?php endif; ?>

<div class="grid">
  <div class="panel">
    <div class="row" style="justify-content:space-between">
      <b style="font-size:12px;letter-spacing:.08em;text-transform:uppercase;color:var(--ts)">Configuration</b>
      <span class="smalltxt"><code>antibot_config.json</code></span>
    </div>

    <form method="post" style="margin-top:10px">
      <input type="hidden" name="action" value="save_config">

      <div style="display:grid;grid-template-columns:1fr 1fr;gap:10px">
        <div>
          <div class="toggle"><label><input type="checkbox" name="filters_enabled" <?= !empty($cfg['filters_enabled'])?'checked':''; ?>> Enable blocking</label></div>

          <hr>

          <div class="toggle"><label><input type="checkbox" name="enable_tz_check" <?= !empty($cfg['enable_tz_check'])?'checked':''; ?>> Wall: timezone check</label></div>
          <div class="toggle"><label><input type="checkbox" name="enable_fp_wall" <?= !empty($cfg['enable_fp_wall'])?'checked':''; ?>> Wall: fingerprint check</label></div>
          <div class="toggle"><label><input type="checkbox" name="enable_anti_headless" <?= !empty($cfg['enable_anti_headless'])?'checked':''; ?>> Wall: anti-headless</label></div>

          <hr>

          <div class="toggle"><label><input type="checkbox" name="use_ipqs" <?= !empty($cfg['use_ipqs'])?'checked':''; ?>> Use IPQS</label></div>
          <div class="toggle"><label><input type="checkbox" name="use_proxycheck" <?= !empty($cfg['use_proxycheck'])?'checked':''; ?>> Use ProxyCheck</label></div>
          <div class="toggle"><label><input type="checkbox" name="use_ipn" <?= !empty($cfg['use_ipn'])?'checked':''; ?>> Use i.pn</label></div>

          <div class="toggle" style="margin-top:8px"><label>
            <input type="checkbox" name="proxycheck_only_proxy_field" <?= !empty($cfg['proxycheck_only_proxy_field'])?'checked':''; ?>>
            ProxyCheck: ONLY “proxy” field (proxy=no is clean)
          </label></div>

          <div class="toggle" style="margin-top:8px"><label>
            <input type="checkbox" name="ipqs_use_fraud_score" <?= !empty($cfg['ipqs_use_fraud_score'])?'checked':''; ?>>
            IPQS: use fraud score threshold
          </label></div>
          <label>IPQS fraud threshold</label>
          <input type="text" name="ipqs_fraud_threshold" value="<?=ab_e((string)($cfg['ipqs_fraud_threshold'] ?? 75))?>">
        </div>

        <div>
          <label>Allowed countries (CSV)</label>
          <input type="text" name="allowed_countries" value="<?=ab_e(implode(',', $cfg['allowed_countries'] ?? []))?>" placeholder="e.g. US,FR">
          <div style="height:6px"></div>

          <label>Blocked countries (CSV)</label>
          <input type="text" name="blocked_countries" value="<?=ab_e(implode(',', $cfg['blocked_countries'] ?? []))?>" placeholder="e.g. RU,CN">

          <div style="height:10px"></div>

          <label>Allowed timezones (selected show here)</label>
          <select name="allowed_timezones[]" multiple>
            <?php foreach(DateTimeZone::listIdentifiers() as $tz): ?>
              <option value="<?=ab_e($tz)?>" <?= in_array($tz, ($cfg['allowed_timezones'] ?? []), true) ? 'selected' : ''; ?>>
                <?=ab_e($tz)?>
              </option>
            <?php endforeach; ?>
          </select>

          <hr>

          <div class="toggle"><label><input type="checkbox" name="use_local_ip_lists" <?= !empty($cfg['use_local_ip_lists'])?'checked':''; ?>> Use /iplist blocking</label></div>

          <label>Enabled iplist files (checkbox)</label>
          <div style="margin-top:6px;display:grid;grid-template-columns:1fr;gap:6px;max-height:210px;overflow:auto;padding:8px;border:1px solid rgba(255,255,255,.08);border-radius:10px;background:#05070b">
            <?php
              $enabled = $cfg['iplist_enabled_files'] ?? [];
              if (!is_array($enabled)) $enabled = [];
              foreach($sources as $k=>$s):
                $file = $s['file'];
                $chk = in_array($file, $enabled, true);
            ?>
              <label style="font-size:12px;color:#e2f3ff">
                <input type="checkbox" name="iplist_enabled_files[]" value="<?=ab_e($file)?>" <?= $chk?'checked':''; ?>>
                <?=ab_e($file)?>
              </label>
            <?php endforeach; ?>
          </div>

          <hr>

          <div class="toggle"><label><input type="checkbox" name="param_gate_enabled" <?= !empty($cfg['param_gate']['enabled'])?'checked':''; ?>> Param Gate enabled</label></div>
          <label>Param name</label>
          <input type="text" name="param_name" value="<?=ab_e($cfg['param_gate']['param_name'] ?? 'id')?>">

          <div style="height:6px"></div>
          <label>Param mode</label>
          <select name="param_mode">
            <?php
              $modes = ['off'=>'Off','exact'=>'Exact match','contains'=>'Contains substring','file'=>'In file (per line)'];
              $curMode = $cfg['param_gate']['mode'] ?? 'off';
              foreach($modes as $k=>$lbl): ?>
              <option value="<?=$k?>" <?= ($curMode===$k)?'selected':''; ?>><?=ab_e($lbl)?></option>
            <?php endforeach; ?>
          </select>

          <div style="height:8px"></div>
          <label>Exact value</label>
          <input type="text" name="param_exact_value" value="<?=ab_e($cfg['param_gate']['exact_value'] ?? '')?>">

          <div style="height:6px"></div>
          <label>Contains substring</label>
          <input type="text" name="param_contains_str" value="<?=ab_e($cfg['param_gate']['contains_str'] ?? '')?>">

          <div style="height:6px"></div>
          <label>Values file</label>
          <input type="text" name="param_values_file" value="<?=ab_e($cfg['param_gate']['values_file'] ?? 'values.txt')?>">

          <div style="height:10px"></div>
          <div class="toggle"><label><input type="checkbox" name="param_remember_pass" <?= !empty($cfg['param_gate']['remember_pass'])?'checked':''; ?>> Remember pass after entry accepted</label></div>
          <label>Entry cookie name</label>
          <input type="text" name="param_cookie_name" value="<?=ab_e($cfg['param_gate']['cookie_name'] ?? 'ab_entry_ok')?>">

          <div style="height:6px"></div>
          <label>Entry cookie days</label>
          <input type="text" name="param_cookie_days" value="<?=ab_e((string)($cfg['param_gate']['cookie_days'] ?? 30))?>">
        </div>
      </div>

      <div class="row" style="justify-content:flex-end;margin-top:10px">
        <button class="btn primary" type="submit">Save</button>
      </div>
    </form>

    <form method="post" style="margin-top:10px">
      <input type="hidden" name="action" value="reset_config">
      <button class="btn danger" type="submit" onclick="return confirm('Reset config to defaults?');">Reset to defaults</button>
    </form>
  </div>

  <div class="panel">
    <div class="row" style="justify-content:space-between">
      <b style="font-size:12px;letter-spacing:.08em;text-transform:uppercase;color:var(--ts)">IPList</b>
      <span class="smalltxt">folder: <code>/iplist/</code></span>
    </div>

    <div class="smalltxt" style="margin-top:8px">
      Update/Download overwrites ALL 9 files and then deletes extra old files in /iplist/.
      If a download fails, do it manually from the URL and save with the exact filename.
    </div>

    <hr>

    <div class="row" style="justify-content:space-between;align-items:flex-end">
      <div class="smalltxt">Action status below</div>
      <div class="row">
        <button class="btn primary" id="btnDownloadAll">Update/Download all (overwrite + cleanup)</button>
      </div>
    </div>

    <div style="margin-top:10px">
      <div class="prog"><div id="bar"></div></div>
      <div class="smalltxt" id="progressText" style="margin-top:6px">Idle</div>
      <ul class="list" id="statusList"></ul>
    </div>

    <hr>

    <div class="smalltxt"><b>Files</b></div>
    <div class="smalltxt" style="margin-top:6px">
      <?php foreach($sources as $k=>$s):
        $path = ab_iplist_path($AB_IPLIST_DIR, $s['file']);
        $exists = is_file($path);
        $size = $exists ? (int)@filesize($path) : 0;
        $mtime = $exists ? (int)@filemtime($path) : 0;
      ?>
        <div style="margin-bottom:6px">
          <code><?=ab_e($s['file'])?></code>
          <span class="smalltxt">— <?=ab_e($s['url'])?></span><br>
          <span class="smalltxt"><?= $exists ? ('Local: yes | size='.$size.' | mtime='.date('Y-m-d H:i:s',$mtime)) : 'Local: NO' ?></span>
        </div>
      <?php endforeach; ?>
    </div>

    <hr>

    <b style="font-size:12px;letter-spacing:.08em;text-transform:uppercase;color:var(--ts)">Quick Block / Whitelist</b>

    <form method="post" style="margin-top:10px;display:flex;gap:10px;align-items:end;flex-wrap:wrap">
      <div style="flex:1;min-width:220px">
        <label>IP or CIDR</label>
        <input type="text" name="ip" placeholder="e.g. 1.2.3.4 or 2a02:6b8::/29 or 1.2.3.0/24">
      </div>
      <button class="btn danger" type="submit" name="action" value="block_ip">Block</button>
      <button class="btn success" type="submit" name="action" value="whitelist">Whitelist</button>
    </form>

    <div class="smalltxt" style="margin-top:8px">
      Whitelist ALWAYS bypasses everything (even if later added to block lists).
    </div>
  </div>
</div>

<div class="panel">
  <div class="row" style="justify-content:space-between">
    <b style="font-size:12px;letter-spacing:.08em;text-transform:uppercase;color:var(--ts)">Visitor Sessions</b>
    <span class="smalltxt"><code>visits.log</code></span>
  </div>

  <div class="row" style="margin-top:10px;gap:8px;flex-wrap:wrap">
    <form method="get" class="row" style="gap:8px;flex-wrap:wrap">
      <input type="hidden" name="admin" value="1">
      <div style="min-width:140px">
        <label>Status</label>
        <select name="status">
          <option value="all"     <?= $statusFilter==='all'?'selected':''; ?>>All</option>
          <option value="allowed" <?= $statusFilter==='allowed'?'selected':''; ?>>Allowed</option>
          <option value="denied"  <?= $statusFilter==='denied'?'selected':''; ?>>Denied</option>
        </select>
      </div>
      <div style="min-width:140px">
        <label>Country</label>
        <input type="text" name="country" value="<?=ab_e($countryFilter)?>" placeholder="US" maxlength="3">
      </div>
      <div style="min-width:140px">
        <label>Device</label>
        <select name="device">
          <option value="all"     <?= $deviceFilter==='all'?'selected':''; ?>>All</option>
          <option value="mobile"  <?= $deviceFilter==='mobile'?'selected':''; ?>>Mobile</option>
          <option value="desktop" <?= $deviceFilter==='desktop'?'selected':''; ?>>Desktop</option>
        </select>
      </div>
      <div style="align-self:end">
        <button class="btn primary" type="submit">Apply</button>
      </div>
    </form>
  </div>

  <div class="tablewrap">
    <table>
      <tr>
        <th>Date</th><th>IP</th><th>Status</th><th>Reason</th>
        <th>Country</th><th>Region</th><th>ISP</th><th>Org</th><th>Fraud</th>
        <th>User Agent</th><th>Actions</th>
      </tr>

      <?php foreach($lines as $line):
        $p = ab_parse_visit_line($line);
        if(!$p['ip']) continue;

        $decisionDisplay = (($p['decision'] ?? '') === 'denied') ? 'denied' : 'allowed';
        $badgeClass = ($decisionDisplay === 'denied') ? 'bad' : 'ok';

        $cc = strtoupper(trim($p['country']));
        $device = ab_device_type_from_ua($p['ua']);

        if($statusFilter === 'allowed' && $decisionDisplay !== 'allowed') continue;
        if($statusFilter === 'denied'  && $decisionDisplay !== 'denied') continue;
        if($countryFilter && $cc !== $countryFilter) continue;
        if($deviceFilter === 'mobile'  && $device !== 'mobile') continue;
        if($deviceFilter === 'desktop' && $device !== 'desktop') continue;

        // FLAGS LOGIC (BACK)
        $flag = (strlen($cc) === 2)
            ? "<img class='flag' src='https://flagsapi.com/{$cc}/shiny/64.png' alt='{$cc}' onerror=\"this.style.display='none'\">"
            : '';
        $jsonPretty = 'No JSON';
        if (!empty($p['json'])) {
            $decoded = json_decode($p['json'], true);
            if (is_array($decoded)) $jsonPretty = json_encode($decoded, JSON_PRETTY_PRINT|JSON_UNESCAPED_SLASHES);
            else $jsonPretty = $p['json'];
        }
      ?>
      <tr>
        <td><?=ab_e($p['datetime'])?></td>
        <td><?=ab_e($p['ip'])?></td>
        <td><span class="badge <?=$badgeClass?>"><?=ab_e($decisionDisplay)?></span></td>
        <td class="reason" title="<?=ab_e($p['reason'])?>"><?=ab_e($p['reason'])?></td>
        <td><?= $flag ?><span><?=ab_e($cc)?></span></td>
        <td><?=ab_e($p['region'])?></td>
        <td><?=ab_e($p['isp'])?></td>
        <td><?=ab_e($p['org'])?></td>
        <td><?=ab_e($p['fraud'])?></td>
        <td class="ua" title="<?=ab_e($p['ua'])?>"><?=ab_e($p['ua'])?></td>
        <td>
          <form method="post" style="display:inline">
            <input type="hidden" name="ip" value="<?=ab_e($p['ip'])?>">
            <button class="btn small danger" type="submit" name="action" value="block_ip">Block</button>
          </form>
          <form method="post" style="display:inline">
            <input type="hidden" name="ip" value="<?=ab_e($p['ip'])?>">
            <button class="btn small success" type="submit" name="action" value="whitelist">Whitelist</button>
          </form>
          <details>
            <summary>Raw</summary>
            <pre><?=ab_e($jsonPretty)?></pre>
          </details>
        </td>
      </tr>
      <?php endforeach; ?>
    </table>
  </div>
</div>

<script>
(async function(){
  const bar = document.getElementById('bar');
  const progressText = document.getElementById('progressText');
  const statusList = document.getElementById('statusList');
  const btn = document.getElementById('btnDownloadAll');

  function setProgress(done, total){
    const pct = total ? Math.round((done/total)*100) : 0;
    bar.style.width = pct + "%";
    progressText.textContent = total ? (`${done}/${total} (${pct}%)`) : 'Idle';
  }
  function addStatus(text){
    const li = document.createElement('li');
    li.textContent = text;
    statusList.appendChild(li);
  }
  function resetUI(){
    statusList.innerHTML = '';
    bar.style.width = '0%';
    progressText.textContent = 'Idle';
  }
  async function api(params){
    const qs = new URLSearchParams(params);
    const url = `<?=ab_e(basename(__FILE__))?>?admin=1&iplist_api=1&` + qs.toString();
    const r = await fetch(url, {credentials:'same-origin'});
    return await r.json();
  }

  btn.addEventListener('click', async function(){
    btn.disabled = true;
    resetUI();

    const list = await api({action:'list'});
    if (!list.ok){
      addStatus('Failed to load iplist sources.');
      btn.disabled = false;
      return;
    }

    const items = list.items || [];
    const total = items.length;
    let done = 0;
    setProgress(done, total);

    // Download ALL and overwrite
    for (const it of items){
      addStatus(`Downloading (overwrite) ${it.file} ...`);
      try{
        const res = await api({action:'download', key: it.key});
        if (res.ok){
          addStatus(`OK: ${it.file} (mtime=${new Date(res.mtime*1000).toISOString().replace('T',' ').replace('Z','')})`);
        } else {
          addStatus(`ERROR: ${it.file} (${res.error || 'download_failed'})`);
          if (res.manual) addStatus(`Manual: ${res.manual}`);
        }
      } catch(e){
        addStatus(`ERROR: ${it.file} (exception)`);
        addStatus(`Manual: download from URL and put in /iplist/${it.file}`);
      }
      done++;
      setProgress(done, total);
    }

    // Cleanup extras
    addStatus('Cleanup: removing old extra files in /iplist/ ...');
    const clean = await api({action:'cleanup'});
    if (clean.ok){
      if ((clean.deleted || []).length){
        addStatus('Deleted extras: ' + clean.deleted.join(', '));
      } else {
        addStatus('No extra files to delete.');
      }
    } else {
      addStatus('Cleanup failed (permissions).');
    }

    addStatus('Done.');
    btn.disabled = false;
  });
})();
</script>

</body>
</html>
<?php
}

/* ---------------------------------------------------------------------
   ROUTER
   --------------------------------------------------------------------- */
if (php_sapi_name() !== 'cli') {
    $selfIsMain = (realpath($_SERVER['SCRIPT_FILENAME'] ?? '') === __FILE__);

    if ($selfIsMain){
        if (isset($_GET['admin'])){
            antibot_admin();
            exit;
        }
        echo "Antibot installed. Admin: ?admin=1";
        exit;
    } else {
        antibot_gate();
        // allowed => parent page continues
    }
}
